// Copyright 2024 Google LLC.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

// Code generated file. DO NOT EDIT.

// Package airquality provides access to the Air Quality API.
//
// For product documentation, see: https://developers.google.com/maps/documentation/air-quality
//
// # Library status
//
// These client libraries are officially supported by Google. However, this
// library is considered complete and is in maintenance mode. This means
// that we will address critical bugs and security issues but will not add
// any new features.
//
// When possible, we recommend using our newer
// [Cloud Client Libraries for Go](https://pkg.go.dev/cloud.google.com/go)
// that are still actively being worked and iterated on.
//
// # Creating a client
//
// Usage example:
//
//	import "google.golang.org/api/airquality/v1"
//	...
//	ctx := context.Background()
//	airqualityService, err := airquality.NewService(ctx)
//
// In this example, Google Application Default Credentials are used for
// authentication. For information on how to create and obtain Application
// Default Credentials, see https://developers.google.com/identity/protocols/application-default-credentials.
//
// # Other authentication options
//
// To use an API key for authentication (note: some APIs do not support API
// keys), use [google.golang.org/api/option.WithAPIKey]:
//
//	airqualityService, err := airquality.NewService(ctx, option.WithAPIKey("AIza..."))
//
// To use an OAuth token (e.g., a user token obtained via a three-legged OAuth
// flow, use [google.golang.org/api/option.WithTokenSource]:
//
//	config := &oauth2.Config{...}
//	// ...
//	token, err := config.Exchange(ctx, ...)
//	airqualityService, err := airquality.NewService(ctx, option.WithTokenSource(config.TokenSource(ctx, token)))
//
// See [google.golang.org/api/option.ClientOption] for details on options.
package airquality // import "google.golang.org/api/airquality/v1"

import (
	"bytes"
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"net/http"
	"net/url"
	"strconv"
	"strings"

	googleapi "google.golang.org/api/googleapi"
	internal "google.golang.org/api/internal"
	gensupport "google.golang.org/api/internal/gensupport"
	option "google.golang.org/api/option"
	internaloption "google.golang.org/api/option/internaloption"
	htransport "google.golang.org/api/transport/http"
)

// Always reference these packages, just in case the auto-generated code
// below doesn't.
var _ = bytes.NewBuffer
var _ = strconv.Itoa
var _ = fmt.Sprintf
var _ = json.NewDecoder
var _ = io.Copy
var _ = url.Parse
var _ = gensupport.MarshalJSON
var _ = googleapi.Version
var _ = errors.New
var _ = strings.Replace
var _ = context.Canceled
var _ = internaloption.WithDefaultEndpoint
var _ = internal.Version

const apiId = "airquality:v1"
const apiName = "airquality"
const apiVersion = "v1"
const basePath = "https://airquality.googleapis.com/"
const basePathTemplate = "https://airquality.UNIVERSE_DOMAIN/"
const mtlsBasePath = "https://airquality.mtls.googleapis.com/"

// OAuth2 scopes used by this API.
const (
	// See, edit, configure, and delete your Google Cloud data and see the email
	// address for your Google Account.
	CloudPlatformScope = "https://www.googleapis.com/auth/cloud-platform"
)

// NewService creates a new Service.
func NewService(ctx context.Context, opts ...option.ClientOption) (*Service, error) {
	scopesOption := internaloption.WithDefaultScopes(
		"https://www.googleapis.com/auth/cloud-platform",
	)
	// NOTE: prepend, so we don't override user-specified scopes.
	opts = append([]option.ClientOption{scopesOption}, opts...)
	opts = append(opts, internaloption.WithDefaultEndpoint(basePath))
	opts = append(opts, internaloption.WithDefaultEndpointTemplate(basePathTemplate))
	opts = append(opts, internaloption.WithDefaultMTLSEndpoint(mtlsBasePath))
	opts = append(opts, internaloption.EnableNewAuthLibrary())
	client, endpoint, err := htransport.NewClient(ctx, opts...)
	if err != nil {
		return nil, err
	}
	s, err := New(client)
	if err != nil {
		return nil, err
	}
	if endpoint != "" {
		s.BasePath = endpoint
	}
	return s, nil
}

// New creates a new Service. It uses the provided http.Client for requests.
//
// Deprecated: please use NewService instead.
// To provide a custom HTTP client, use option.WithHTTPClient.
// If you are using google.golang.org/api/googleapis/transport.APIKey, use option.WithAPIKey with NewService instead.
func New(client *http.Client) (*Service, error) {
	if client == nil {
		return nil, errors.New("client is nil")
	}
	s := &Service{client: client, BasePath: basePath}
	s.CurrentConditions = NewCurrentConditionsService(s)
	s.Forecast = NewForecastService(s)
	s.History = NewHistoryService(s)
	s.MapTypes = NewMapTypesService(s)
	return s, nil
}

type Service struct {
	client    *http.Client
	BasePath  string // API endpoint base URL
	UserAgent string // optional additional User-Agent fragment

	CurrentConditions *CurrentConditionsService

	Forecast *ForecastService

	History *HistoryService

	MapTypes *MapTypesService
}

func (s *Service) userAgent() string {
	if s.UserAgent == "" {
		return googleapi.UserAgent
	}
	return googleapi.UserAgent + " " + s.UserAgent
}

func NewCurrentConditionsService(s *Service) *CurrentConditionsService {
	rs := &CurrentConditionsService{s: s}
	return rs
}

type CurrentConditionsService struct {
	s *Service
}

func NewForecastService(s *Service) *ForecastService {
	rs := &ForecastService{s: s}
	return rs
}

type ForecastService struct {
	s *Service
}

func NewHistoryService(s *Service) *HistoryService {
	rs := &HistoryService{s: s}
	return rs
}

type HistoryService struct {
	s *Service
}

func NewMapTypesService(s *Service) *MapTypesService {
	rs := &MapTypesService{s: s}
	rs.HeatmapTiles = NewMapTypesHeatmapTilesService(s)
	return rs
}

type MapTypesService struct {
	s *Service

	HeatmapTiles *MapTypesHeatmapTilesService
}

func NewMapTypesHeatmapTilesService(s *Service) *MapTypesHeatmapTilesService {
	rs := &MapTypesHeatmapTilesService{s: s}
	return rs
}

type MapTypesHeatmapTilesService struct {
	s *Service
}

// AdditionalInfo: The emission sources and health effects of a given
// pollutant.
type AdditionalInfo struct {
	// Effects: Text representing the pollutant's main health effects.
	Effects string `json:"effects,omitempty"`
	// Sources: Text representing the pollutant's main emission sources.
	Sources string `json:"sources,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Effects") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Effects") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s AdditionalInfo) MarshalJSON() ([]byte, error) {
	type NoMethod AdditionalInfo
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// AirQualityIndex: The basic object for representing different air quality
// metrics. When brought together, these metrics provide a snapshot about the
// current air quality conditions. There are multiple indexes in the world
// serving different purposes and groups interested in measuring different
// aspects of air quality.
type AirQualityIndex struct {
	// Aqi:  The index's numeric score. Examples: 10, 100. The value is not
	// normalized and should only be interpreted in the context of its related
	// air-quality index. For non-numeric indexes, this field will not be returned.
	// Note: This field should be used for calculations, graph display, etc. For
	// displaying the index score, you should use the AQI display field.
	Aqi int64 `json:"aqi,omitempty"`
	// AqiDisplay: Textual representation of the index numeric score, that may
	// include prefix or suffix symbols, which usually represents the worst index
	// score. Example: >100 or 10+. Note: This field should be used when you want
	// to display the index score. For non-numeric indexes, this field is empty.
	AqiDisplay string `json:"aqiDisplay,omitempty"`
	// Category: Textual classification of the index numeric score interpretation.
	// For example: "Excellent air quality".
	Category string `json:"category,omitempty"`
	// Code: The index's code. This field represents the index for programming
	// purposes by using snake case instead of spaces. Examples: "uaqi",
	// "fra_atmo".
	Code string `json:"code,omitempty"`
	// Color: The color used to represent the AQI numeric score.
	Color *Color `json:"color,omitempty"`
	// DisplayName: A human readable representation of the index name. Example:
	// "AQI (US)"
	DisplayName string `json:"displayName,omitempty"`
	// DominantPollutant: The chemical symbol of the dominant pollutant. For
	// example: "CO".
	DominantPollutant string `json:"dominantPollutant,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Aqi") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Aqi") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s AirQualityIndex) MarshalJSON() ([]byte, error) {
	type NoMethod AirQualityIndex
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Color: Represents a color in the RGBA color space. This representation is
// designed for simplicity of conversion to and from color representations in
// various languages over compactness. For example, the fields of this
// representation can be trivially provided to the constructor of
// `java.awt.Color` in Java; it can also be trivially provided to UIColor's
// `+colorWithRed:green:blue:alpha` method in iOS; and, with just a little
// work, it can be easily formatted into a CSS `rgba()` string in JavaScript.
// This reference page doesn't have information about the absolute color space
// that should be used to interpret the RGB value—for example, sRGB, Adobe
// RGB, DCI-P3, and BT.2020. By default, applications should assume the sRGB
// color space. When color equality needs to be decided, implementations,
// unless documented otherwise, treat two colors as equal if all their red,
// green, blue, and alpha values each differ by at most `1e-5`. Example (Java):
// import com.google.type.Color; // ... public static java.awt.Color
// fromProto(Color protocolor) { float alpha = protocolor.hasAlpha() ?
// protocolor.getAlpha().getValue() : 1.0; return new java.awt.Color(
// protocolor.getRed(), protocolor.getGreen(), protocolor.getBlue(), alpha); }
// public static Color toProto(java.awt.Color color) { float red = (float)
// color.getRed(); float green = (float) color.getGreen(); float blue = (float)
// color.getBlue(); float denominator = 255.0; Color.Builder resultBuilder =
// Color .newBuilder() .setRed(red / denominator) .setGreen(green /
// denominator) .setBlue(blue / denominator); int alpha = color.getAlpha(); if
// (alpha != 255) { result.setAlpha( FloatValue .newBuilder()
// .setValue(((float) alpha) / denominator) .build()); } return
// resultBuilder.build(); } // ... Example (iOS / Obj-C): // ... static
// UIColor* fromProto(Color* protocolor) { float red = [protocolor red]; float
// green = [protocolor green]; float blue = [protocolor blue]; FloatValue*
// alpha_wrapper = [protocolor alpha]; float alpha = 1.0; if (alpha_wrapper !=
// nil) { alpha = [alpha_wrapper value]; } return [UIColor colorWithRed:red
// green:green blue:blue alpha:alpha]; } static Color* toProto(UIColor* color)
// { CGFloat red, green, blue, alpha; if (![color getRed:&red green:&green
// blue:&blue alpha:&alpha]) { return nil; } Color* result = [[Color alloc]
// init]; [result setRed:red]; [result setGreen:green]; [result setBlue:blue];
// if (alpha <= 0.9999) { [result setAlpha:floatWrapperWithValue(alpha)]; }
// [result autorelease]; return result; } // ... Example (JavaScript): // ...
// var protoToCssColor = function(rgb_color) { var redFrac = rgb_color.red ||
// 0.0; var greenFrac = rgb_color.green || 0.0; var blueFrac = rgb_color.blue
// || 0.0; var red = Math.floor(redFrac * 255); var green =
// Math.floor(greenFrac * 255); var blue = Math.floor(blueFrac * 255); if
// (!('alpha' in rgb_color)) { return rgbToCssColor(red, green, blue); } var
// alphaFrac = rgb_color.alpha.value || 0.0; var rgbParams = [red, green,
// blue].join(','); return ['rgba(', rgbParams, ',', alphaFrac, ')'].join(”);
// }; var rgbToCssColor = function(red, green, blue) { var rgbNumber = new
// Number((red << 16) | (green << 8) | blue); var hexString =
// rgbNumber.toString(16); var missingZeros = 6 - hexString.length; var
// resultBuilder = ['#']; for (var i = 0; i < missingZeros; i++) {
// resultBuilder.push('0'); } resultBuilder.push(hexString); return
// resultBuilder.join(”); }; // ...
type Color struct {
	// Alpha: The fraction of this color that should be applied to the pixel. That
	// is, the final pixel color is defined by the equation: `pixel color = alpha *
	// (this color) + (1.0 - alpha) * (background color)` This means that a value
	// of 1.0 corresponds to a solid color, whereas a value of 0.0 corresponds to a
	// completely transparent color. This uses a wrapper message rather than a
	// simple float scalar so that it is possible to distinguish between a default
	// value and the value being unset. If omitted, this color object is rendered
	// as a solid color (as if the alpha value had been explicitly given a value of
	// 1.0).
	Alpha float64 `json:"alpha,omitempty"`
	// Blue: The amount of blue in the color as a value in the interval [0, 1].
	Blue float64 `json:"blue,omitempty"`
	// Green: The amount of green in the color as a value in the interval [0, 1].
	Green float64 `json:"green,omitempty"`
	// Red: The amount of red in the color as a value in the interval [0, 1].
	Red float64 `json:"red,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Alpha") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Alpha") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Color) MarshalJSON() ([]byte, error) {
	type NoMethod Color
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *Color) UnmarshalJSON(data []byte) error {
	type NoMethod Color
	var s1 struct {
		Alpha gensupport.JSONFloat64 `json:"alpha"`
		Blue  gensupport.JSONFloat64 `json:"blue"`
		Green gensupport.JSONFloat64 `json:"green"`
		Red   gensupport.JSONFloat64 `json:"red"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.Alpha = float64(s1.Alpha)
	s.Blue = float64(s1.Blue)
	s.Green = float64(s1.Green)
	s.Red = float64(s1.Red)
	return nil
}

// Concentration: The concentration of a given pollutant in the air.
type Concentration struct {
	// Units: Units for measuring this pollutant concentration.
	//
	// Possible values:
	//   "UNIT_UNSPECIFIED" - Unspecified concentration unit.
	//   "PARTS_PER_BILLION" - The ppb (parts per billion) concentration unit.
	//   "MICROGRAMS_PER_CUBIC_METER" - The "µg/m^3" (micrograms per cubic meter)
	// concentration unit.
	Units string `json:"units,omitempty"`
	// Value: Value of the pollutant concentration.
	Value float64 `json:"value,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Units") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Units") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Concentration) MarshalJSON() ([]byte, error) {
	type NoMethod Concentration
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *Concentration) UnmarshalJSON(data []byte) error {
	type NoMethod Concentration
	var s1 struct {
		Value gensupport.JSONFloat64 `json:"value"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.Value = float64(s1.Value)
	return nil
}

// CustomLocalAqi: Expresses a 'country/region to AQI' relationship. Pairs a
// country/region with a desired AQI so that air quality data that is required
// for that country/region will be displayed according to the chosen AQI.
type CustomLocalAqi struct {
	// Aqi: The AQI to associate the country/region with. Value should be a valid
	// index (/maps/documentation/air-quality/laqis) code.
	Aqi string `json:"aqi,omitempty"`
	// RegionCode: The country/region requiring the custom AQI. Value should be
	// provided using ISO 3166-1 alpha-2
	// (https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2) code.
	RegionCode string `json:"regionCode,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Aqi") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Aqi") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s CustomLocalAqi) MarshalJSON() ([]byte, error) {
	type NoMethod CustomLocalAqi
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// HealthRecommendations: Health recommendations for different population
// groups in a free text format. The recommendations are derived from their
// associated air quality conditions.
type HealthRecommendations struct {
	// Athletes: Sports and other strenuous outdoor activities.
	Athletes string `json:"athletes,omitempty"`
	// Children: Younger populations including children, toddlers, and babies.
	Children string `json:"children,omitempty"`
	// Elderly: Retirees and people older than the general population.
	Elderly string `json:"elderly,omitempty"`
	// GeneralPopulation: No specific sensitivities.
	GeneralPopulation string `json:"generalPopulation,omitempty"`
	// HeartDiseasePopulation: Heart and circulatory system diseases.
	HeartDiseasePopulation string `json:"heartDiseasePopulation,omitempty"`
	// LungDiseasePopulation: Respiratory related problems and asthma suffers.
	LungDiseasePopulation string `json:"lungDiseasePopulation,omitempty"`
	// PregnantWomen: Women at all stages of pregnancy.
	PregnantWomen string `json:"pregnantWomen,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Athletes") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Athletes") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s HealthRecommendations) MarshalJSON() ([]byte, error) {
	type NoMethod HealthRecommendations
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// HourInfo: Contains the air quality information for each hour in the
// requested range. For example, if the request is for 48 hours of history
// there will be 48 elements of hourly info.
type HourInfo struct {
	// DateTime: A rounded down timestamp indicating the time the data refers to in
	// RFC3339 UTC "Zulu" format, with nanosecond resolution and up to nine
	// fractional digits. For example: "2014-10-02T15:00:00Z".
	DateTime string `json:"dateTime,omitempty"`
	// HealthRecommendations: Health advice and recommended actions related to the
	// reported air quality conditions. Recommendations are tailored differently
	// for populations at risk, groups with greater sensitivities to pollutants,
	// and the general population.
	HealthRecommendations *HealthRecommendations `json:"healthRecommendations,omitempty"`
	// Indexes: Based on the request parameters, this list will include (up to) two
	// air quality indexes: - Universal AQI. Will be returned if the universalAqi
	// boolean is set to true. - Local AQI. Will be returned if the LOCAL_AQI extra
	// computation is specified.
	Indexes []*AirQualityIndex `json:"indexes,omitempty"`
	// Pollutants: A list of pollutants affecting the location specified in the
	// request. Note: This field will be returned only for requests that specified
	// one or more of the following extra computations: POLLUTANT_ADDITIONAL_INFO,
	// DOMINANT_POLLUTANT_CONCENTRATION, POLLUTANT_CONCENTRATION.
	Pollutants []*Pollutant `json:"pollutants,omitempty"`
	// ForceSendFields is a list of field names (e.g. "DateTime") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "DateTime") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s HourInfo) MarshalJSON() ([]byte, error) {
	type NoMethod HourInfo
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// HourlyForecast: Contains the air quality information for each hour in the
// requested range. For example, if the request is for 48 hours of forecast
// there will be 48 elements of hourly forecasts.
type HourlyForecast struct {
	// DateTime: A rounded down timestamp indicating the time (hour) the data
	// refers to in RFC3339 UTC "Zulu" format. For example: "2014-10-02T15:00:00Z".
	DateTime string `json:"dateTime,omitempty"`
	// HealthRecommendations: Health advice and recommended actions related to the
	// reported air quality conditions. Recommendations are tailored differently
	// for populations at risk, groups with greater sensitivities to pollutants,
	// and the general population.
	HealthRecommendations *HealthRecommendations `json:"healthRecommendations,omitempty"`
	// Indexes: Based on the request parameters, this list will include (up to) two
	// air quality indexes: - Universal AQI. Will be returned if the
	// `universal_aqi` boolean is set to true. - Local AQI. Will be returned if the
	// LOCAL_AQI extra computation is specified.
	Indexes []*AirQualityIndex `json:"indexes,omitempty"`
	// Pollutants: A list of pollutants affecting the location specified in the
	// request. Note: This field will be returned only for requests that specified
	// one or more of the following extra computations: POLLUTANT_ADDITIONAL_INFO,
	// DOMINANT_POLLUTANT_CONCENTRATION, POLLUTANT_CONCENTRATION.
	Pollutants []*Pollutant `json:"pollutants,omitempty"`
	// ForceSendFields is a list of field names (e.g. "DateTime") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "DateTime") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s HourlyForecast) MarshalJSON() ([]byte, error) {
	type NoMethod HourlyForecast
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// HttpBody: Message that represents an arbitrary HTTP body. It should only be
// used for payload formats that can't be represented as JSON, such as raw
// binary or an HTML page. This message can be used both in streaming and
// non-streaming API methods in the request as well as the response. It can be
// used as a top-level request field, which is convenient if one wants to
// extract parameters from either the URL or HTTP template into the request
// fields and also want access to the raw HTTP body. Example: message
// GetResourceRequest { // A unique request id. string request_id = 1; // The
// raw HTTP body is bound to this field. google.api.HttpBody http_body = 2; }
// service ResourceService { rpc GetResource(GetResourceRequest) returns
// (google.api.HttpBody); rpc UpdateResource(google.api.HttpBody) returns
// (google.protobuf.Empty); } Example with streaming methods: service
// CaldavService { rpc GetCalendar(stream google.api.HttpBody) returns (stream
// google.api.HttpBody); rpc UpdateCalendar(stream google.api.HttpBody) returns
// (stream google.api.HttpBody); } Use of this type only changes how the
// request and response bodies are handled, all other features will continue to
// work unchanged.
type HttpBody struct {
	// ContentType: The HTTP Content-Type header value specifying the content type
	// of the body.
	ContentType string `json:"contentType,omitempty"`
	// Data: The HTTP request/response body as raw binary.
	Data string `json:"data,omitempty"`
	// Extensions: Application specific response metadata. Must be set in the first
	// response for streaming APIs.
	Extensions []googleapi.RawMessage `json:"extensions,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "ContentType") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ContentType") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s HttpBody) MarshalJSON() ([]byte, error) {
	type NoMethod HttpBody
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Interval: Represents a time interval, encoded as a Timestamp start
// (inclusive) and a Timestamp end (exclusive). The start must be less than or
// equal to the end. When the start equals the end, the interval is empty
// (matches no time). When both start and end are unspecified, the interval
// matches any time.
type Interval struct {
	// EndTime: Optional. Exclusive end of the interval. If specified, a Timestamp
	// matching this interval will have to be before the end.
	EndTime string `json:"endTime,omitempty"`
	// StartTime: Optional. Inclusive start of the interval. If specified, a
	// Timestamp matching this interval will have to be the same or after the
	// start.
	StartTime string `json:"startTime,omitempty"`
	// ForceSendFields is a list of field names (e.g. "EndTime") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "EndTime") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Interval) MarshalJSON() ([]byte, error) {
	type NoMethod Interval
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// LatLng: An object that represents a latitude/longitude pair. This is
// expressed as a pair of doubles to represent degrees latitude and degrees
// longitude. Unless specified otherwise, this object must conform to the WGS84
// standard. Values must be within normalized ranges.
type LatLng struct {
	// Latitude: The latitude in degrees. It must be in the range [-90.0, +90.0].
	Latitude float64 `json:"latitude,omitempty"`
	// Longitude: The longitude in degrees. It must be in the range [-180.0,
	// +180.0].
	Longitude float64 `json:"longitude,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Latitude") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Latitude") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s LatLng) MarshalJSON() ([]byte, error) {
	type NoMethod LatLng
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *LatLng) UnmarshalJSON(data []byte) error {
	type NoMethod LatLng
	var s1 struct {
		Latitude  gensupport.JSONFloat64 `json:"latitude"`
		Longitude gensupport.JSONFloat64 `json:"longitude"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.Latitude = float64(s1.Latitude)
	s.Longitude = float64(s1.Longitude)
	return nil
}

// LookupCurrentConditionsRequest: The request definition of the air quality
// current conditions.
type LookupCurrentConditionsRequest struct {
	// CustomLocalAqis: Optional. Expresses a 'country/region to AQI' relationship.
	// Pairs a country/region with a desired AQI so that air quality data that is
	// required for that country/region will be displayed according to the chosen
	// AQI. This parameter can be used to specify a non-default AQI for a given
	// country, for example, to get the US EPA index for Canada rather than the
	// default index for Canada.
	CustomLocalAqis []*CustomLocalAqi `json:"customLocalAqis,omitempty"`
	// ExtraComputations: Optional. Additional features that can be optionally
	// enabled. Specifying extra computations will result in the relevant elements
	// and fields to be returned in the response.
	//
	// Possible values:
	//   "EXTRA_COMPUTATION_UNSPECIFIED" - The default value. The server ignores it
	// if it is passed as a parameter.
	//   "LOCAL_AQI" - Determines whether to include the local (national) AQI of
	// the requested location (country) in the response. If specified, the response
	// will contain an 'air_quality_index' data structure with all the relevant
	// data on the location's local AQI.
	//   "HEALTH_RECOMMENDATIONS" - Determines whether the response will include
	// the health advice and recommended actions for the current AQI conditions.
	// The recommendations are tailored for the general population and six
	// populations at risk groups with greater sensitivities to pollutants than the
	// general population. If specified, the `health_recommendations` field will be
	// populated in the response when the relevant data is available.
	//   "POLLUTANT_ADDITIONAL_INFO" - Determines whether to include in the
	// response the additional information of each pollutant. If specified, each
	// air quality index object contained in the 'indexes' field response will
	// include an `additional_info` field when the data is available.
	//   "DOMINANT_POLLUTANT_CONCENTRATION" - Determines whether the response would
	// include the concentrations of the dominant pollutants measured according to
	// global and/or local indexes. If the request specified both the global AQI
	// and the local AQI, there may be up to two pollutant codes returned. If
	// specified, the dominant pollutant object contained in the 'pollutants' list
	// will include a `concentration` field when the data is available.
	//   "POLLUTANT_CONCENTRATION" - Determines whether the response would include
	// the concentrations of all pollutants with available measurements according
	// to global and/or local indexes. If specified, each pollutant object
	// contained in the 'pollutants' field in the response will include a
	// `concentration` field when the data is available.
	ExtraComputations []string `json:"extraComputations,omitempty"`
	// LanguageCode: Optional. Allows the client to choose the language for the
	// response. If data cannot be provided for that language the API uses the
	// closest match. Allowed values rely on the IETF standard. Default value is
	// en.
	LanguageCode string `json:"languageCode,omitempty"`
	// Location: Required. The longitude and latitude from which the API looks for
	// air quality current conditions data.
	Location *LatLng `json:"location,omitempty"`
	// UaqiColorPalette: Optional. Determines the color palette used for data
	// provided by the 'Universal Air Quality Index' (UAQI). This color palette is
	// relevant just for UAQI, other AQIs have a predetermined color palette that
	// can't be controlled.
	//
	// Possible values:
	//   "COLOR_PALETTE_UNSPECIFIED" - The default value. Ignored if passed as a
	// parameter.
	//   "RED_GREEN" - Determines whether to use a red/green palette.
	//   "INDIGO_PERSIAN_DARK" - Determines whether to use a indigo/persian palette
	// (dark theme).
	//   "INDIGO_PERSIAN_LIGHT" - Determines whether to use a indigo/persian
	// palette (light theme).
	UaqiColorPalette string `json:"uaqiColorPalette,omitempty"`
	// UniversalAqi: Optional. If set to true, the Universal AQI will be included
	// in the 'indexes' field of the response. Default value is true.
	UniversalAqi bool `json:"universalAqi,omitempty"`
	// ForceSendFields is a list of field names (e.g. "CustomLocalAqis") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CustomLocalAqis") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s LookupCurrentConditionsRequest) MarshalJSON() ([]byte, error) {
	type NoMethod LookupCurrentConditionsRequest
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

type LookupCurrentConditionsResponse struct {
	// DateTime: A rounded down timestamp in RFC3339 UTC "Zulu" format, with
	// nanosecond resolution and up to nine fractional digits. For example:
	// "2014-10-02T15:00:00Z".
	DateTime string `json:"dateTime,omitempty"`
	// HealthRecommendations: Health advice and recommended actions related to the
	// reported air quality conditions. Recommendations are tailored differently
	// for populations at risk, groups with greater sensitivities to pollutants,
	// and the general population.
	HealthRecommendations *HealthRecommendations `json:"healthRecommendations,omitempty"`
	// Indexes: Based on the request parameters, this list will include (up to) two
	// air quality indexes: - Universal AQI. Will be returned if the universalAqi
	// boolean is set to true. - Local AQI. Will be returned if the LOCAL_AQI extra
	// computation is specified.
	Indexes []*AirQualityIndex `json:"indexes,omitempty"`
	// Pollutants: A list of pollutants affecting the location specified in the
	// request. Note: This field will be returned only for requests that specified
	// one or more of the following extra computations: POLLUTANT_ADDITIONAL_INFO,
	// DOMINANT_POLLUTANT_CONCENTRATION, POLLUTANT_CONCENTRATION.
	Pollutants []*Pollutant `json:"pollutants,omitempty"`
	// RegionCode: The ISO_3166-1 alpha-2 code of the country/region corresponding
	// to the location provided in the request. This field might be omitted from
	// the response if the location provided in the request resides in a disputed
	// territory.
	RegionCode string `json:"regionCode,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "DateTime") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "DateTime") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s LookupCurrentConditionsResponse) MarshalJSON() ([]byte, error) {
	type NoMethod LookupCurrentConditionsResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// LookupForecastRequest: The request object of the air quality forecast API.
type LookupForecastRequest struct {
	// CustomLocalAqis: Optional. Expresses a 'country/region to AQI' relationship.
	// Pairs a country/region with a desired AQI so that air quality data that is
	// required for that country/region will be displayed according to the chosen
	// AQI. This parameter can be used to specify a non-default AQI for a given
	// country, for example, to get the US EPA index for Canada rather than the
	// default index for Canada.
	CustomLocalAqis []*CustomLocalAqi `json:"customLocalAqis,omitempty"`
	// DateTime: A timestamp for which to return the data for a specific point in
	// time. The timestamp is rounded to the previous exact hour. Note: this will
	// return hourly data for the requested timestamp only (i.e. a single hourly
	// info element). For example, a request sent where the date_time parameter is
	// set to 2023-01-03T11:05:49Z will be rounded down to 2023-01-03T11:00:00Z.
	DateTime string `json:"dateTime,omitempty"`
	// ExtraComputations: Optional. Additional features that can be optionally
	// enabled. Specifying extra computations will result in the relevant elements
	// and fields to be returned in the response.
	//
	// Possible values:
	//   "EXTRA_COMPUTATION_UNSPECIFIED" - The default value. The server ignores it
	// if it is passed as a parameter.
	//   "LOCAL_AQI" - Determines whether to include the local (national) AQI of
	// the requested location (country) in the response. If specified, the response
	// will contain an 'air_quality_index' data structure with all the relevant
	// data on the location's local AQI.
	//   "HEALTH_RECOMMENDATIONS" - Determines whether the response will include
	// the health advice and recommended actions for the current AQI conditions.
	// The recommendations are tailored for the general population and six
	// populations at risk groups with greater sensitivities to pollutants than the
	// general population. If specified, the `health_recommendations` field will be
	// populated in the response when the relevant data is available.
	//   "POLLUTANT_ADDITIONAL_INFO" - Determines whether to include in the
	// response the additional information of each pollutant. If specified, each
	// air quality index object contained in the 'indexes' field response will
	// include an `additional_info` field when the data is available.
	//   "DOMINANT_POLLUTANT_CONCENTRATION" - Determines whether the response would
	// include the concentrations of the dominant pollutants measured according to
	// global and/or local indexes. If the request specified both the global AQI
	// and the local AQI, there may be up to two pollutant codes returned. If
	// specified, the dominant pollutant object contained in the 'pollutants' list
	// will include a `concentration` field when the data is available.
	//   "POLLUTANT_CONCENTRATION" - Determines whether the response would include
	// the concentrations of all pollutants with available measurements according
	// to global and/or local indexes. If specified, each pollutant object
	// contained in the 'pollutants' field in the response will include a
	// `concentration` field when the data is available.
	ExtraComputations []string `json:"extraComputations,omitempty"`
	// LanguageCode: Optional. Allows the client to choose the language for the
	// response. If data cannot be provided for that language the API uses the
	// closest match. Allowed values rely on the IETF standard (default = 'en').
	LanguageCode string `json:"languageCode,omitempty"`
	// Location: Required. The latitude and longitude for which the API looks for
	// air quality data.
	Location *LatLng `json:"location,omitempty"`
	// PageSize: Optional. The maximum number of hourly info records to return per
	// page (default = 24).
	PageSize int64 `json:"pageSize,omitempty"`
	// PageToken: Optional. A page token received from a previous forecast call. It
	// is used to retrieve the subsequent page.
	PageToken string `json:"pageToken,omitempty"`
	// Period: Indicates the start and end period for which to get the forecast
	// data. The timestamp is rounded to the previous exact hour.
	Period *Interval `json:"period,omitempty"`
	// UaqiColorPalette: Optional. Determines the color palette used for data
	// provided by the 'Universal Air Quality Index' (UAQI). This color palette is
	// relevant just for UAQI, other AQIs have a predetermined color palette that
	// can't be controlled.
	//
	// Possible values:
	//   "COLOR_PALETTE_UNSPECIFIED" - The default value. Ignored if passed as a
	// parameter.
	//   "RED_GREEN" - Determines whether to use a red/green palette.
	//   "INDIGO_PERSIAN_DARK" - Determines whether to use a indigo/persian palette
	// (dark theme).
	//   "INDIGO_PERSIAN_LIGHT" - Determines whether to use a indigo/persian
	// palette (light theme).
	UaqiColorPalette string `json:"uaqiColorPalette,omitempty"`
	// UniversalAqi: Optional. If set to true, the Universal AQI will be included
	// in the 'indexes' field of the response (default = true).
	UniversalAqi bool `json:"universalAqi,omitempty"`
	// ForceSendFields is a list of field names (e.g. "CustomLocalAqis") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CustomLocalAqis") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s LookupForecastRequest) MarshalJSON() ([]byte, error) {
	type NoMethod LookupForecastRequest
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// LookupForecastResponse: The response object of the air quality forecast API.
type LookupForecastResponse struct {
	// HourlyForecasts: Optional. Contains the air quality information for each
	// hour in the requested range. For example, if the request is for 48 hours of
	// forecast there will be 48 elements of hourly forecasts.
	HourlyForecasts []*HourlyForecast `json:"hourlyForecasts,omitempty"`
	// NextPageToken: Optional. The token to retrieve the next page.
	NextPageToken string `json:"nextPageToken,omitempty"`
	// RegionCode: Optional. The ISO_3166-1 alpha-2 code of the country/region
	// corresponding to the location provided in the request. This field might be
	// omitted from the response if the location provided in the request resides in
	// a disputed territory.
	RegionCode string `json:"regionCode,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "HourlyForecasts") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "HourlyForecasts") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s LookupForecastResponse) MarshalJSON() ([]byte, error) {
	type NoMethod LookupForecastResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// LookupHistoryRequest: The request object of the air quality history API.
type LookupHistoryRequest struct {
	// CustomLocalAqis: Optional. Expresses a 'country/region to AQI' relationship.
	// Pairs a country/region with a desired AQI so that air quality data that is
	// required for that country/region will be displayed according to the chosen
	// AQI. This parameter can be used to specify a non-default AQI for a given
	// country, for example, to get the US EPA index for Canada rather than the
	// default index for Canada.
	CustomLocalAqis []*CustomLocalAqi `json:"customLocalAqis,omitempty"`
	// DateTime: A timestamp for which to return historical data. The timestamp is
	// rounded to the previous exact hour. Note: this will return hourly data for
	// the requested timestamp only (i.e. a single hourly info element). For
	// example, a request sent where the dateTime parameter is set to
	// 2023-01-03T11:05:49Z will be rounded down to 2023-01-03T11:00:00Z. A
	// timestamp in RFC3339 UTC "Zulu" format, with nanosecond resolution and up to
	// nine fractional digits. Examples: "2014-10-02T15:01:23Z" and
	// "2014-10-02T15:01:23.045123456Z".
	DateTime string `json:"dateTime,omitempty"`
	// ExtraComputations: Optional. Additional features that can be optionally
	// enabled. Specifying extra computations will result in the relevant elements
	// and fields to be returned in the response.
	//
	// Possible values:
	//   "EXTRA_COMPUTATION_UNSPECIFIED" - The default value. The server ignores it
	// if it is passed as a parameter.
	//   "LOCAL_AQI" - Determines whether to include the local (national) AQI of
	// the requested location (country) in the response. If specified, the response
	// will contain an 'air_quality_index' data structure with all the relevant
	// data on the location's local AQI.
	//   "HEALTH_RECOMMENDATIONS" - Determines whether the response will include
	// the health advice and recommended actions for the current AQI conditions.
	// The recommendations are tailored for the general population and six
	// populations at risk groups with greater sensitivities to pollutants than the
	// general population. If specified, the `health_recommendations` field will be
	// populated in the response when the relevant data is available.
	//   "POLLUTANT_ADDITIONAL_INFO" - Determines whether to include in the
	// response the additional information of each pollutant. If specified, each
	// air quality index object contained in the 'indexes' field response will
	// include an `additional_info` field when the data is available.
	//   "DOMINANT_POLLUTANT_CONCENTRATION" - Determines whether the response would
	// include the concentrations of the dominant pollutants measured according to
	// global and/or local indexes. If the request specified both the global AQI
	// and the local AQI, there may be up to two pollutant codes returned. If
	// specified, the dominant pollutant object contained in the 'pollutants' list
	// will include a `concentration` field when the data is available.
	//   "POLLUTANT_CONCENTRATION" - Determines whether the response would include
	// the concentrations of all pollutants with available measurements according
	// to global and/or local indexes. If specified, each pollutant object
	// contained in the 'pollutants' field in the response will include a
	// `concentration` field when the data is available.
	ExtraComputations []string `json:"extraComputations,omitempty"`
	// Hours: Number from 1 to 720 that indicates the hours range for the request.
	// For example: A value of 48 will yield data from the last 48 hours.
	Hours int64 `json:"hours,omitempty"`
	// LanguageCode: Optional. Allows the client to choose the language for the
	// response. If data cannot be provided for that language the API uses the
	// closest match. Allowed values rely on the IETF standard. Default value is
	// en.
	LanguageCode string `json:"languageCode,omitempty"`
	// Location: Required. The latitude and longitude for which the API looks for
	// air quality history data.
	Location *LatLng `json:"location,omitempty"`
	// PageSize: Optional. The maximum number of hourly info records to return per
	// page. The default is 72 and the max value is 168 (7 days of data).
	PageSize int64 `json:"pageSize,omitempty"`
	// PageToken: Optional. A page token received from a previous history call. It
	// is used to retrieve the subsequent page. Note that when providing a value
	// for this parameter all other parameters provided must match the call that
	// provided the page token (the previous call).
	PageToken string `json:"pageToken,omitempty"`
	// Period: Indicates the start and end period for which to get the historical
	// data. The timestamp is rounded to the previous exact hour.
	Period *Interval `json:"period,omitempty"`
	// UaqiColorPalette: Optional. Determines the color palette used for data
	// provided by the 'Universal Air Quality Index' (UAQI). This color palette is
	// relevant just for UAQI, other AQIs have a predetermined color palette that
	// can't be controlled.
	//
	// Possible values:
	//   "COLOR_PALETTE_UNSPECIFIED" - The default value. Ignored if passed as a
	// parameter.
	//   "RED_GREEN" - Determines whether to use a red/green palette.
	//   "INDIGO_PERSIAN_DARK" - Determines whether to use a indigo/persian palette
	// (dark theme).
	//   "INDIGO_PERSIAN_LIGHT" - Determines whether to use a indigo/persian
	// palette (light theme).
	UaqiColorPalette string `json:"uaqiColorPalette,omitempty"`
	// UniversalAqi: Optional. If set to true, the Universal AQI will be included
	// in the 'indexes' field of the response. Default value is true.
	UniversalAqi bool `json:"universalAqi,omitempty"`
	// ForceSendFields is a list of field names (e.g. "CustomLocalAqis") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CustomLocalAqis") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s LookupHistoryRequest) MarshalJSON() ([]byte, error) {
	type NoMethod LookupHistoryRequest
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

type LookupHistoryResponse struct {
	// HoursInfo: Optional. Contains the air quality information for each hour in
	// the requested range. For example, if the request is for 48 hours of history
	// there will be 48 elements of hourly info.
	HoursInfo []*HourInfo `json:"hoursInfo,omitempty"`
	// NextPageToken: Optional. The token to retrieve the next page.
	NextPageToken string `json:"nextPageToken,omitempty"`
	// RegionCode: Optional. The ISO_3166-1 alpha-2 code of the country/region
	// corresponding to the location provided in the request. This field might be
	// omitted from the response if the location provided in the request resides in
	// a disputed territory.
	RegionCode string `json:"regionCode,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "HoursInfo") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "HoursInfo") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s LookupHistoryResponse) MarshalJSON() ([]byte, error) {
	type NoMethod LookupHistoryResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Pollutant: Data regarding an air quality pollutant.
type Pollutant struct {
	// AdditionalInfo: Additional information about the pollutant.
	AdditionalInfo *AdditionalInfo `json:"additionalInfo,omitempty"`
	// Code: The pollutant's code name (for example, "so2"). For a list of
	// supported pollutant codes, see Reported pollutants
	// (/maps/documentation/air-quality/pollutants#reported_pollutants).
	Code string `json:"code,omitempty"`
	// Concentration: The pollutant's concentration level measured by one of the
	// standard air pollutation measure units.
	Concentration *Concentration `json:"concentration,omitempty"`
	// DisplayName: The pollutant's display name. For example: "NOx".
	DisplayName string `json:"displayName,omitempty"`
	// FullName: The pollutant's full name. For chemical compounds, this is the
	// IUPAC name. Example: "Sulfur Dioxide". For more information about the IUPAC
	// names table, see https://iupac.org/what-we-do/periodic-table-of-elements/.
	FullName string `json:"fullName,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AdditionalInfo") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AdditionalInfo") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Pollutant) MarshalJSON() ([]byte, error) {
	type NoMethod Pollutant
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

type CurrentConditionsLookupCall struct {
	s                              *Service
	lookupcurrentconditionsrequest *LookupCurrentConditionsRequest
	urlParams_                     gensupport.URLParams
	ctx_                           context.Context
	header_                        http.Header
}

// Lookup: The Current Conditions endpoint provides hourly air quality
// information in more than 100 countries, up to a 500 x 500 meters resolution.
// Includes over 70 local indexes and global air quality index and categories.
func (r *CurrentConditionsService) Lookup(lookupcurrentconditionsrequest *LookupCurrentConditionsRequest) *CurrentConditionsLookupCall {
	c := &CurrentConditionsLookupCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.lookupcurrentconditionsrequest = lookupcurrentconditionsrequest
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *CurrentConditionsLookupCall) Fields(s ...googleapi.Field) *CurrentConditionsLookupCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *CurrentConditionsLookupCall) Context(ctx context.Context) *CurrentConditionsLookupCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *CurrentConditionsLookupCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *CurrentConditionsLookupCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	var body io.Reader = nil
	body, err := googleapi.WithoutDataWrapper.JSONReader(c.lookupcurrentconditionsrequest)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/currentConditions:lookup")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "airquality.currentConditions.lookup" call.
// Any non-2xx status code is an error. Response headers are in either
// *LookupCurrentConditionsResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *CurrentConditionsLookupCall) Do(opts ...googleapi.CallOption) (*LookupCurrentConditionsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &LookupCurrentConditionsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

type ForecastLookupCall struct {
	s                     *Service
	lookupforecastrequest *LookupForecastRequest
	urlParams_            gensupport.URLParams
	ctx_                  context.Context
	header_               http.Header
}

// Lookup: Returns air quality forecast for a specific location for a given
// time range.
func (r *ForecastService) Lookup(lookupforecastrequest *LookupForecastRequest) *ForecastLookupCall {
	c := &ForecastLookupCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.lookupforecastrequest = lookupforecastrequest
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ForecastLookupCall) Fields(s ...googleapi.Field) *ForecastLookupCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ForecastLookupCall) Context(ctx context.Context) *ForecastLookupCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ForecastLookupCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ForecastLookupCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	var body io.Reader = nil
	body, err := googleapi.WithoutDataWrapper.JSONReader(c.lookupforecastrequest)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/forecast:lookup")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "airquality.forecast.lookup" call.
// Any non-2xx status code is an error. Response headers are in either
// *LookupForecastResponse.ServerResponse.Header or (if a response was returned
// at all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *ForecastLookupCall) Do(opts ...googleapi.CallOption) (*LookupForecastResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &LookupForecastResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *ForecastLookupCall) Pages(ctx context.Context, f func(*LookupForecastResponse) error) error {
	c.ctx_ = ctx
	defer func(pt string) { c.lookupforecastrequest.PageToken = pt }(c.lookupforecastrequest.PageToken)
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.lookupforecastrequest.PageToken = x.NextPageToken
	}
}

type HistoryLookupCall struct {
	s                    *Service
	lookuphistoryrequest *LookupHistoryRequest
	urlParams_           gensupport.URLParams
	ctx_                 context.Context
	header_              http.Header
}

// Lookup: Returns air quality history for a specific location for a given time
// range.
func (r *HistoryService) Lookup(lookuphistoryrequest *LookupHistoryRequest) *HistoryLookupCall {
	c := &HistoryLookupCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.lookuphistoryrequest = lookuphistoryrequest
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *HistoryLookupCall) Fields(s ...googleapi.Field) *HistoryLookupCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *HistoryLookupCall) Context(ctx context.Context) *HistoryLookupCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *HistoryLookupCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *HistoryLookupCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	var body io.Reader = nil
	body, err := googleapi.WithoutDataWrapper.JSONReader(c.lookuphistoryrequest)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/history:lookup")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "airquality.history.lookup" call.
// Any non-2xx status code is an error. Response headers are in either
// *LookupHistoryResponse.ServerResponse.Header or (if a response was returned
// at all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *HistoryLookupCall) Do(opts ...googleapi.CallOption) (*LookupHistoryResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &LookupHistoryResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *HistoryLookupCall) Pages(ctx context.Context, f func(*LookupHistoryResponse) error) error {
	c.ctx_ = ctx
	defer func(pt string) { c.lookuphistoryrequest.PageToken = pt }(c.lookuphistoryrequest.PageToken)
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.lookuphistoryrequest.PageToken = x.NextPageToken
	}
}

type MapTypesHeatmapTilesLookupHeatmapTileCall struct {
	s            *Service
	mapType      string
	zoom         int64
	x            int64
	y            int64
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// LookupHeatmapTile: Returns a bytes array containing the data of the tile PNG
// image.
//
//   - mapType: The type of the air quality heatmap. Defines the pollutant that
//     the map will graphically represent. Allowed values: - UAQI_RED_GREEN
//     (UAQI, red-green palette) - UAQI_INDIGO_PERSIAN (UAQI, indigo-persian
//     palette) - PM25_INDIGO_PERSIAN - GBR_DEFRA - DEU_UBA - CAN_EC - FRA_ATMO -
//     US_AQI.
//   - x: Defines the east-west point in the requested tile.
//   - y: Defines the north-south point in the requested tile.
//   - zoom: The map's zoom level. Defines how large or small the contents of a
//     map appear in a map view. Zoom level 0 is the entire world in a single
//     tile. Zoom level 1 is the entire world in 4 tiles. Zoom level 2 is the
//     entire world in 16 tiles. Zoom level 16 is the entire world in 65,536
//     tiles. Allowed values: 0-16.
func (r *MapTypesHeatmapTilesService) LookupHeatmapTile(mapType string, zoom int64, x int64, y int64) *MapTypesHeatmapTilesLookupHeatmapTileCall {
	c := &MapTypesHeatmapTilesLookupHeatmapTileCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.mapType = mapType
	c.zoom = zoom
	c.x = x
	c.y = y
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *MapTypesHeatmapTilesLookupHeatmapTileCall) Fields(s ...googleapi.Field) *MapTypesHeatmapTilesLookupHeatmapTileCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *MapTypesHeatmapTilesLookupHeatmapTileCall) IfNoneMatch(entityTag string) *MapTypesHeatmapTilesLookupHeatmapTileCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *MapTypesHeatmapTilesLookupHeatmapTileCall) Context(ctx context.Context) *MapTypesHeatmapTilesLookupHeatmapTileCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *MapTypesHeatmapTilesLookupHeatmapTileCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *MapTypesHeatmapTilesLookupHeatmapTileCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/mapTypes/{mapType}/heatmapTiles/{zoom}/{x}/{y}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"mapType": c.mapType,
		"zoom":    strconv.FormatInt(c.zoom, 10),
		"x":       strconv.FormatInt(c.x, 10),
		"y":       strconv.FormatInt(c.y, 10),
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "airquality.mapTypes.heatmapTiles.lookupHeatmapTile" call.
// Any non-2xx status code is an error. Response headers are in either
// *HttpBody.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *MapTypesHeatmapTilesLookupHeatmapTileCall) Do(opts ...googleapi.CallOption) (*HttpBody, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &HttpBody{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}
