// Copyright 2025 Google LLC
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package samreg

import (
	"errors"
	"slices"
	"strings"
	"testing"
)

func TestNewUserV(t *testing.T) {
	tests := []struct {
		name        string
		buffer      []byte
		wantErr     bool
		wantErrText string
	}{
		{
			name:   "valid_userV_succeeds",
			buffer: []byte(strings.Repeat("\x00", 0xCC)),
		},
		{
			name:        "invalid_userV_returns_error",
			buffer:      []byte("A"),
			wantErr:     true,
			wantErrText: "unexpected EOF",
		},
	}

	for _, tc := range tests {
		t.Run(tc.name, func(t *testing.T) {
			_, err := newUserV(tc.buffer, "irrelevant")
			if (err != nil) != tc.wantErr {
				t.Errorf("NewUserV(...): Unexpected error: %v", err)
			}

			if tc.wantErr {
				if !strings.Contains(err.Error(), tc.wantErrText) {
					t.Errorf("NewUserV(...): Invalid error: got: %v, want: %v", err, tc.wantErrText)
				}
			}
		})
	}
}

func TestUserVRead(t *testing.T) {
	tests := []struct {
		name     string
		buffer   []byte
		offset   uint32
		size     uint32
		wantData []byte
		wantErr  error
	}{
		{
			name:     "valid_userV_succeeds",
			buffer:   []byte(strings.Repeat("A", 0xCC) + "BCDE"),
			offset:   0,
			size:     4,
			wantData: []byte("BCDE"),
		},
		{
			name:    "out_of_bounds_read_returns_error",
			buffer:  []byte(strings.Repeat("A", 0xCC)),
			offset:  100,
			size:    10,
			wantErr: errReadOutOfBounds,
		},
	}

	for _, tc := range tests {
		t.Run(tc.name, func(t *testing.T) {
			userV, err := newUserV(tc.buffer, "irrelevant")
			if err != nil {
				t.Fatalf("Failed to create userV for tests: %v", err)
			}

			got, gotErr := userV.read(tc.offset, tc.size)
			if !errors.Is(gotErr, tc.wantErr) {
				t.Errorf("Read(): got error: %v, want: %v", gotErr, tc.wantErr)
			}

			if tc.wantErr != nil {
				return
			}

			if !slices.Equal(got, tc.wantData) {
				t.Errorf("Read(): got %v, want %v", got, tc.wantData)
			}
		})
	}
}

func TestUserVUsername(t *testing.T) {
	tests := []struct {
		name         string
		buffer       []byte
		wantUsername string
		wantErr      bool
		wantErrText  string
	}{
		{
			name:         "valid_structure_returns_username",
			buffer:       []byte(strings.Repeat("\x00", 16) + "\x08" + strings.Repeat("\x00", 187) + "\x42\x00\x43\x00\x44\x00\x45\x00"),
			wantUsername: "BCDE",
		},
		{
			name:        "out_of_bounds_read_returns_error",
			buffer:      []byte(strings.Repeat("\xFF", 0xCC)),
			wantErr:     true,
			wantErrText: errReadOutOfBounds.Error(),
		},
	}

	for _, tc := range tests {
		t.Run(tc.name, func(t *testing.T) {
			userV, err := newUserV(tc.buffer, "irrelevant")
			if err != nil {
				t.Fatalf("Failed to create userV for tests: %v", err)
			}

			got, gotErr := userV.Username()
			if (gotErr != nil) != tc.wantErr {
				t.Errorf("Username(): unexpected error: %v", gotErr)
			}

			if tc.wantErr {
				if !strings.Contains(gotErr.Error(), tc.wantErrText) {
					t.Errorf("Username(): got error: %v, want: %v", gotErr, tc.wantErrText)
				}

				return
			}

			if strings.Compare(got, tc.wantUsername) != 0 {
				t.Errorf("Username(): got %v, want %v", got, tc.wantUsername)
			}
		})
	}
}

func TestUserVEncryptedHashes(t *testing.T) {
	tests := []struct {
		name    string
		buffer  []byte
		wantLM  []byte
		wantNT  []byte
		wantErr error
	}{
		{
			name:   "valid_structure_returns_hashes",
			buffer: []byte("\x00\x00\x00\x00\xd4\x00\x00\x00\x02\x00\x01\x00\xd4\x00\x00\x00\x0c\x00\x00\x00\x00\x00\x00\x00\xe0\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xe0\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xe0\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xe0\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xe0\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xe0\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xe0\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xe0\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xe0\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xe0\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xe0\x00\x00\x00\x08\x00\x00\x00\x01\x00\x00\x00\xe8\x00\x00\x00\x14\x00\x00\x00\x00\x00\x00\x00\xfc\x00\x00\x00\x14\x00\x00\x00\x00\x00\x00\x00\x10\x01\x00\x00\x04\x00\x00\x00\x00\x00\x00\x00\x14\x01\x00\x00\x04\x00\x00\x00\x00\x00\x00\x00\x01\x00\x14\x80\xb4\x00\x00\x00\xc4\x00\x00\x00\x14\x00\x00\x00\x44\x00\x00\x00\x02\x00\x30\x00\x02\x00\x00\x00\x02\xc0\x14\x00\x44\x00\x05\x01\x01\x01\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00\x02\xc0\x14\x00\xff\xff\x1f\x00\x01\x01\x00\x00\x00\x00\x00\x05\x07\x00\x00\x00\x02\x00\x70\x00\x04\x00\x00\x00\x00\x00\x14\x00\x5b\x03\x02\x00\x01\x01\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x18\x00\xff\x07\x0f\x00\x01\x02\x00\x00\x00\x00\x00\x05\x20\x00\x00\x00\x20\x02\x00\x00\x00\x00\x18\x00\xff\x07\x0f\x00\x01\x02\x00\x00\x00\x00\x00\x05\x20\x00\x00\x00\x24\x02\x00\x00\x00\x00\x24\x00\x44\x00\x02\x00\x01\x05\x00\x00\x00\x00\x00\x05\x15\x00\x00\x00\xee\x9d\xd5\xc6\x43\xed\x3f\x53\x2e\xc8\x5e\x0a\xe9\x03\x00\x00\x01\x02\x00\x00\x00\x00\x00\x05\x20\x00\x00\x00\x20\x02\x00\x00\x01\x02\x00\x00\x00\x00\x00\x05\x20\x00\x00\x00\x20\x02\x00\x00\x6c\x00\x6d\x00\x75\x00\x73\x00\x65\x00\x72\x00\x01\x02\x00\x00\x07\x00\x00\x00\x03\x00\x01\x00\x0c\x4b\x53\xdf\x15\xed\xf9\xeb\x41\x05\xfe\xca\x48\x02\x70\x55\x03\x00\x01\x00\x72\x10\xad\x76\x18\xf1\x1f\x38\x49\x24\x6d\x13\x38\x15\x8b\x03\x03\x00\x01\x00\x03\x00\x01\x00"),
			wantLM: []byte("\x03\x00\x01\x00\x0c\x4b\x53\xdf\x15\xed\xf9\xeb\x41\x05\xfe\xca\x48\x02\x70\x55"),
			wantNT: []byte("\x03\x00\x01\x00\x72\x10\xad\x76\x18\xf1\x1f\x38\x49\x24\x6d\x13\x38\x15\x8b\x03"),
		},
		{
			name:   "when_only_NTLM_returns_only_NTLM",
			buffer: []byte("\x00\x00\x00\x00\xbc\x00\x00\x00\x02\x00\x01\x00\xbc\x00\x00\x00\x1a\x00\x00\x00\x00\x00\x00\x00\xd8\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\x00\x00\x00\x6c\x00\x00\x00\x00\x00\x00\x00\x44\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x44\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x44\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x44\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x44\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x44\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x44\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x44\x01\x00\x00\x15\x00\x00\x00\xa8\x00\x00\x00\x5c\x01\x00\x00\x08\x00\x00\x00\x01\x00\x00\x00\x64\x01\x00\x00\x04\x00\x00\x00\x00\x00\x00\x00\x68\x01\x00\x00\x14\x00\x00\x00\x00\x00\x00\x00\x7c\x01\x00\x00\x04\x00\x00\x00\x00\x00\x00\x00\x80\x01\x00\x00\x04\x00\x00\x00\x00\x00\x00\x00\x01\x00\x14\x80\x9c\x00\x00\x00\xac\x00\x00\x00\x14\x00\x00\x00\x44\x00\x00\x00\x02\x00\x30\x00\x02\x00\x00\x00\x02\xc0\x14\x00\x44\x00\x05\x01\x01\x01\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00\x02\xc0\x14\x00\xff\xff\x1f\x00\x01\x01\x00\x00\x00\x00\x00\x05\x07\x00\x00\x00\x02\x00\x58\x00\x03\x00\x00\x00\x00\x00\x14\x00\x5b\x03\x02\x00\x01\x01\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x18\x00\xff\x07\x0f\x00\x01\x02\x00\x00\x00\x00\x00\x05\x20\x00\x00\x00\x20\x02\x00\x00\x00\x00\x24\x00\x44\x00\x02\x00\x01\x05\x00\x00\x00\x00\x00\x05\x15\x00\x00\x00\xee\x9d\xd5\xc6\x43\xed\x3f\x53\x2e\xc8\x5e\x0a\xf4\x01\x00\x00\x01\x02\x00\x00\x00\x00\x00\x05\x20\x00\x00\x00\x20\x02\x00\x00\x01\x02\x00\x00\x00\x00\x00\x05\x20\x00\x00\x00\x20\x02\x00\x00\x41\x00\x64\x00\x6d\x00\x69\x00\x6e\x00\x69\x00\x73\x00\x74\x00\x72\x00\x61\x00\x74\x00\x6f\x00\x72\x00\x64\x00\x42\x00\x75\x00\x69\x00\x6c\x00\x74\x00\x2d\x00\x69\x00\x6e\x00\x20\x00\x61\x00\x63\x00\x63\x00\x6f\x00\x75\x00\x6e\x00\x74\x00\x20\x00\x66\x00\x6f\x00\x72\x00\x20\x00\x61\x00\x64\x00\x6d\x00\x69\x00\x6e\x00\x69\x00\x73\x00\x74\x00\x65\x00\x72\x00\x69\x00\x6e\x00\x67\x00\x20\x00\x74\x00\x68\x00\x65\x00\x20\x00\x63\x00\x6f\x00\x6d\x00\x70\x00\x75\x00\x74\x00\x65\x00\x72\x00\x2f\x00\x64\x00\x6f\x00\x6d\x00\x61\x00\x69\x00\x6e\x00\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x00\x01\x00\x01\x02\x00\x00\x07\x00\x00\x00\x03\x00\x01\x00\x03\x00\x01\x00\xed\x92\x87\x92\x78\x3b\x69\x2c\x21\x37\x49\xbc\xdb\xe3\x1a\xf5\x03\x00\x01\x00\x03\x00\x01\x00"),
			wantLM: []byte("\x03\x00\x01\x00"),
			wantNT: []byte("\x03\x00\x01\x00\xed\x92\x87\x92\x78\x3b\x69\x2c\x21\x37\x49\xbc\xdb\xe3\x1a\xf5"),
		},
		{
			name:    "missing_hash_info_returns_error",
			buffer:  []byte(strings.Repeat("\x00", 0xCC)),
			wantErr: errNoHashInfoFound,
		},
		{
			name:    "out_of_bounds_LM_read_returns_error",
			buffer:  []byte(strings.Repeat("\x00", 0x9C) + strings.Repeat("\xFF", 8) + "\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00" + strings.Repeat("\x00", 0xCC)),
			wantErr: errReadOutOfBounds,
		},
		{
			name:    "out_of_bounds_NT_read_returns_error",
			buffer:  []byte(strings.Repeat("\xFF", 0xCC)),
			wantErr: errReadOutOfBounds,
		},
	}

	for _, tc := range tests {
		t.Run(tc.name, func(t *testing.T) {
			userV, err := newUserV(tc.buffer, "irrelevant")
			if err != nil {
				t.Fatalf("Failed to create userV for tests: %v", err)
			}

			gotLM, gotNT, gotErr := userV.EncryptedHashes()
			if !errors.Is(gotErr, tc.wantErr) {
				t.Errorf("EncryptedHashes(): unexpected error, got: %v, want: %v", gotErr, tc.wantErr)
			}

			if tc.wantErr != nil {
				return
			}

			if !slices.Equal(gotLM, tc.wantLM) {
				t.Errorf("EncryptedHashes(): got LM hash: %v, want: %v", gotLM, tc.wantLM)
			}

			if !slices.Equal(gotNT, tc.wantNT) {
				t.Errorf("EncryptedHashes(): got NT hash: %v, want: %v", gotNT, tc.wantNT)
			}
		})
	}
}
